/*
 * Copyright (C) 2010 Olivier PARISOT <parisot_olivier@yahoo.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.doopyon.ravanelab.core.domain;

import javax.persistence.*;


/**
 *
 * Entity representing League.
 * This class is responsible for the domain object related
 * business logic for League. Properties and associations are
 * implemented in the generated base class {@link org.doopyon.ravanelab.core.domain.LeagueBase}.
 */
@Entity
@Table(name = "LEAGUE")
@org.hibernate.annotations.Cache(usage = org.hibernate.annotations.CacheConcurrencyStrategy.NONSTRICT_READ_WRITE)
public class League extends LeagueBase 
{
	//
	// Static fields
	//
	
	/** Serial version UID. */
	private static final long serialVersionUID=316660703971339841L;


	//
	// Constructors
	//
	
	/**
	 * Constructor.
	 */
    protected League()
    {
    	super();
    }    

    /**
     * Constructor.
     * @param leagueName the league name
     */
    public League(String leagueName) 
    {
        super(leagueName);
    }

    
    //
    // Instance methods
    //
    
    /**
     * Get the league season corresponding to a given league season year.
     * @return the league season corresponding to a given league season year, null otherwise
     */
	public final LeagueSeason getLeagueSeasonForYear(int currentYear) 
	{
		for (LeagueSeason ls:getLeagueSeasons())
		{
			if (ls.getSeasonYear()==currentYear) return ls;
		}
		return null;	
	}
	
    /**
     * Get the first league season, according to the season year.
     * @return the first league season, according to the season year, null otherwise
     */
	public final LeagueSeason getFirstLeagueSeason() 
	{
		LeagueSeason firstLeagueSeason=null;
		int minYear=Integer.MAX_VALUE;
		for (LeagueSeason ls:getLeagueSeasons())
		{			
			if (ls.getSeasonYear()<minYear)
			{
				minYear=ls.getSeasonYear();
				firstLeagueSeason=ls;
			}
		}
		return firstLeagueSeason;	
	}
	
    /**
     * Get the last league season, according to the season year.
     * @return the last league season, according to the season year, null otherwise
     */
	public final LeagueSeason getLastLeagueSeason() 
	{
		LeagueSeason lastLeagueSeason=null;
		int maxYear=Integer.MIN_VALUE;
		for (LeagueSeason ls:getLeagueSeasons())
		{			
			if (ls.getSeasonYear()>maxYear)
			{
				maxYear=ls.getSeasonYear();
				lastLeagueSeason=ls;
			}
		}
		return lastLeagueSeason;	
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String toString()
	{
		final StringBuilder sb=new StringBuilder("League: ").append(getLeagueName()).append("\n\n");		
		for (LeagueSeason ls:getLeagueSeasons()) sb.append(ls).append("\n");					
		return sb.toString();
	}
}
