/*
 * Copyright (C) 2010 Olivier PARISOT <parisot_olivier@yahoo.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.doopyon.ravanelab.core.domain;

import javax.persistence.*;
import java.util.*;


/**
 *
 * Entity representing LeagueSeason.
 * This class is responsible for the domain object related
 * business logic for LeagueSeason. Properties and associations are
 * implemented in the generated base class {@link org.doopyon.ravanelab.core.domain.LeagueSeasonBase}.
 */
@Entity
@Table(name = "LEAGUESEASON")
@org.hibernate.annotations.Cache(usage = org.hibernate.annotations.CacheConcurrencyStrategy.NONSTRICT_READ_WRITE)
public class LeagueSeason extends LeagueSeasonBase 
{
	//
	// Static fields
	//
		
	/**	Serial version UID. */
	private static final long serialVersionUID = -120247450998584241L;
	   
    
	//
	// Constructors
	//

	/**
	 * Constructor.
	 */
    protected LeagueSeason()
    {
    	super();
    }

    /**
     * Constructor.
     * @param seasonYear
     */
    public LeagueSeason(final int seasonYear) 
    {
        super(seasonYear);
    }	
    
	
    //
    // Instance methods
    //
    
    /**
     * Get a match week by its order.
     * @return the match week if found, null otherwise
     */
	public final MatchWeek getMatchWeek(final int order) 
	{
		if (order<=0) throw new IllegalArgumentException("No match week with order=0");
		
		for (MatchWeek mw:getMatchWeeks())
		{
			if (mw.getMatchWeekOrder()==order) return mw;
		}
		return null;
	}
	
	/**
	 * Get the list of the referenced teams in this league season.
	 * @return the list of the referenced teams in this league season
	 */
	public final Set<Team> getTeams()
	{
		final Set<Team> l=new HashSet<Team>();
		for (MatchWeek mw:getMatchWeeks()) l.addAll(mw.getTeams());		
		return l;
	}
	
	/**
	 * 
	 * @return
	 */
	public final Set<Integer> getAvailableMathWeeksOrders()
	{
		final Set<Integer> l=new HashSet<Integer>();
		for (MatchWeek mw:getMatchWeeks()) l.add(mw.getMatchWeekOrder());		
		return l;		
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String toString()
	{
		final StringBuilder sb=new StringBuilder("Season year: ").append(getSeasonYear()).append("\n\n");		
		for (MatchWeek mw:getMatchWeeks()) sb.append(mw).append("\n");					
		return sb.toString();
	}

	/**
	 * 
	 * @return
	 */
	public final boolean isComplete() 
	{
		int nbMatchs=0;
		for (MatchWeek mw:getMatchWeeks()) nbMatchs+=mw.getMatchs().size();
		final int nbTeams=getTeams().size();
		final int expectedMatchsCount=(nbTeams-1)*nbTeams;
		return nbMatchs==expectedMatchsCount;
	}
}
