/*
 * Copyright (C) 2010 Olivier PARISOT <parisot_olivier@yahoo.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.doopyon.ravanelab.core.domain;

import java.io.*;
import java.util.*;


/**
 * Value object representing LeagueSeasonRanking.
 * This class is responsible for the domain object related
 * business logic for LeagueSeasonRanking. Properties and associations are
 * implemented in the generated base class {@link org.doopyon.ravanelab.core.domain.LeagueSeasonRankingBase}.
 * 
 * @author Olivier PARISOT
 */
public class LeagueSeasonRanking extends LeagueSeasonRankingBase 
{
	//
	// Static fields
	//
	
	/**	Serial version UID. */
	private static final long serialVersionUID=-722021504902242313L;

    
	//
	// Constructor
	//
	
	/**
	 * Constructor.
	 */
	public LeagueSeasonRanking(final int seasonYear,final int matchWeekOrder)
	{
		super(seasonYear,matchWeekOrder);
	}
	
	
    //
    // Instance fields
    //
    
    /**
     * 
     */
	public final LeagueSeasonRankingByTeam getLeagueSeasonRankingByTeam(final Team team) 
	{
		for (LeagueSeasonRankingByTeam lsrt:getLeagueSeasonRankingByTeams())
		{
			if (lsrt.getRankedTeam().equals(team)) return lsrt;
		}		
		return null;
	}
	
	/**
	 * 
	 */
	private final int getIndexOfTeam(final Team team,final Comparator<LeagueSeasonRankingByTeam> comparator)
	{
		final List<LeagueSeasonRankingByTeam> lsrbt=getSortedList(comparator);
		final int size=lsrbt.size();
		for (int i=0;i<size;i++)
		{
			if (lsrbt.get(i).getRankedTeam().equals(team)) return i;
		}
		return -1;
	}
	
	/**
	 * 
	 */
	private final List<LeagueSeasonRankingByTeam> getSortedList(final Comparator<LeagueSeasonRankingByTeam> comparator)
	{
		final List<LeagueSeasonRankingByTeam> col=new ArrayList<LeagueSeasonRankingByTeam>();
		col.addAll(getLeagueSeasonRankingByTeams());
		Collections.sort(col,comparator);
		return col;
	}
	
	/**
	 * 
	 */
	public final int getRankOfTeamByPoints(final Team team)
	{
		if (getMatchWeekOrder()==0) return 0;
		
		final int idx=getIndexOfTeam(team,new LeagueSeasonRankingByTeamComparatorByPoints());
		if (idx<0) throw new IllegalStateException("Team '"+team+"' not found in ranking -> "+this);
		return 1+idx;
	}

	/**
	 * 
	 */
	public final int getRankOfTeamByGoalsFor(final Team team)
	{
		if (getMatchWeekOrder()==0) return 0;		
		
		final int idx=getIndexOfTeam(team,new LeagueSeasonRankingByTeamComparatorByGoalsFor());
		if (idx<0) throw new IllegalStateException("Team '"+team+"' not found in ranking -> "+this);
		return 1+idx;
	}
	
	/**
	 * 
	 */
	public final int getRankOfTeamByGoalsAgainst(final Team team)
	{
		if (getMatchWeekOrder()==0) return 0;
		
		final int idx=getIndexOfTeam(team,new LeagueSeasonRankingByTeamComparatorByGoalsAgainst());
		if (idx<0) throw new IllegalStateException("Team '"+team+"' not found in ranking -> "+this);
		return 1+idx;
	}	
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String toString()
	{		
		final List<LeagueSeasonRankingByTeam> lsrbt=getSortedList(new LeagueSeasonRankingByTeamComparatorByPoints());
		final int size=lsrbt.size();
		final StringBuilder sb=new StringBuilder();
		for (int i=0;i<size;i++) sb.append(lsrbt.get(i)).append("\n");		
		return sb.toString();
	}
	
	
	//
	// Static classes
	//
	
	/**
	 * Comparator by points.
	 */	
	private static class LeagueSeasonRankingByTeamComparatorByPoints implements Comparator<LeagueSeasonRankingByTeam>,Serializable 
	{
		//
		// Static fields
		//
		
		/** Serial version UID. */
		private static final long serialVersionUID=1425047203742803522L;

		
		//
		// Instance methods
		//
		
		/**
		 * {@inheritDoc}
		 */
		@Override
		public final int compare(final LeagueSeasonRankingByTeam firstTeamRanking,final LeagueSeasonRankingByTeam secondTeamRanking) 
	    {
			final int firstTeamPoint=firstTeamRanking.getCountOfPoints();
			final int secondTeamPoint=secondTeamRanking.getCountOfPoints();
			if (firstTeamPoint==secondTeamPoint)
			{
				final int firstTeamGoalAverage=firstTeamRanking.getCountOfGoalsFor()-firstTeamRanking.getCountOfGoalsAgainst();
				final int secondTeamGoalAverage=secondTeamRanking.getCountOfGoalsFor()-secondTeamRanking.getCountOfGoalsAgainst();
				return secondTeamGoalAverage-firstTeamGoalAverage;
			}
			else return secondTeamPoint-firstTeamPoint;
	    }
	}
	
	/**
	 * Comparator by 'goals for'.
	 */	
	private static class LeagueSeasonRankingByTeamComparatorByGoalsFor implements Comparator<LeagueSeasonRankingByTeam>,Serializable 
	{
		//
		// Static fields
		//
		
		/** Serial version UID. */
		private static final long serialVersionUID=1425047203742803522L;

		
		//
		// Instance methods
		//
		
		/**
		 * {@inheritDoc}
		 */
		@Override
		public final int compare(final LeagueSeasonRankingByTeam firstTeamRanking,final LeagueSeasonRankingByTeam secondTeamRanking) 
	    {
			return secondTeamRanking.getCountOfGoalsFor()-firstTeamRanking.getCountOfGoalsFor();
	    }
	}
	
	/**
	 * Comparator by 'goals against'.
	 */	
	private static class LeagueSeasonRankingByTeamComparatorByGoalsAgainst implements Comparator<LeagueSeasonRankingByTeam>,Serializable 
	{
		//
		// Static fields
		//
		
		/** Serial version UID. */
		private static final long serialVersionUID=1425047203742803522L;

		
		//
		// Instance methods
		//
		
		/**
		 * {@inheritDoc}
		 */
		@Override
		public final int compare(final LeagueSeasonRankingByTeam firstTeamRanking,final LeagueSeasonRankingByTeam secondTeamRanking) 
	    {
			return -secondTeamRanking.getCountOfGoalsAgainst()+firstTeamRanking.getCountOfGoalsAgainst();
	    }
	}	
}
