/*
 * Copyright (C) 2010 Olivier PARISOT <parisot_olivier@yahoo.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.doopyon.ravanelab.core.serviceimpl;

import java.util.*;
import org.doopyon.ravanelab.core.domain.*;
import org.doopyon.ravanelab.core.serviceapi.*;
import org.springframework.stereotype.*;


/**
 * Implementation of LeagueRankingService.
 * 
 * @author Olivier PARISOT
 */
@Service(LeagueSeasonRankingServiceImpl.BEAN_ID)
public class LeagueSeasonRankingServiceImpl extends LeagueSeasonRankingServiceImplBase 
{
    //
    // Instance methods
    //
    
    /**
     * {@inheritDoc}
     */
	@Override
    public final LeagueSeasonRanking getLeagueSeasonRanking(final League l,final int seasonYear,final int currentMatchWeekOrder) 
    {
		/* check arguments */
		if (l==null) throw new IllegalArgumentException("Null league!");
		final LeagueSeason leagueSeason=l.getLeagueSeasonForYear(seasonYear);
		if (leagueSeason==null) throw new IllegalArgumentException("League season '"+l.getLeagueName()+"','"+seasonYear+"' not found!");    	
    	if (currentMatchWeekOrder<0) throw new IllegalArgumentException("Negative match week order!");
    	
    	/* build empty ranking and temporary map */
    	final Map<Team,LeagueSeasonRankingByTeam> mapRankingByTeam=new HashMap<Team,LeagueSeasonRankingByTeam>();    	
		final LeagueSeasonRanking emptyRanking=new LeagueSeasonRanking(seasonYear,currentMatchWeekOrder);
		for (Team t:leagueSeason.getTeams())
		{
			final LeagueSeasonRankingByTeam lsrt=new LeagueSeasonRankingByTeam(0,0,0,0,0,0,t);			
			emptyRanking.addLeagueSeasonRankingByTeam(lsrt);
			mapRankingByTeam.put(t,lsrt);
		}    	 
    	
		/* return if needed */
    	if (currentMatchWeekOrder==0) return emptyRanking;    	
    		
    	/* visit all match weeks */
    	final MatchEvaluatorService matchEvaluatorService=getMatchEvaluatorService();
    	for (MatchWeek mw:leagueSeason.getMatchWeeks())
    	{
    		if (mw.getMatchWeekOrder()<=currentMatchWeekOrder)
    		{
    			for (Match m:mw.getMatchs())
    			{
    				/* evaluate match */    				
					final MatchEvaluation matchEval=matchEvaluatorService.getMatchEvaluation(m);    				
    				if (matchEval.equals(MatchEvaluation.NOTPLAYED)) continue;
    				
    				/* get current rankings in temporary map */
    				LeagueSeasonRankingByTeam currentRankingHomeTeam=mapRankingByTeam.get(m.getHomeTeam());
    				LeagueSeasonRankingByTeam currentRankingAwayTeam=mapRankingByTeam.get(m.getAwayTeam());
					
    				/* update rankings with scores of matchs */
    				currentRankingHomeTeam=currentRankingHomeTeam.withAddingCountOfGoalsFor(m.getScore().getHomePoints())
    															 .withAddingCountOfGoalsAgainst(m.getScore().getAwayPoints());
    				currentRankingAwayTeam=currentRankingAwayTeam.withAddingCountOfGoalsFor(m.getScore().getAwayPoints())
					 											 .withAddingCountOfGoalsAgainst(m.getScore().getHomePoints());
    				    				
    				/* update rankings with count of points, count of won/lost/draw matchs */
    				if (matchEval.equals(MatchEvaluation.RESULT_1))
    				{
    					currentRankingHomeTeam=currentRankingHomeTeam.withAddingCountOfWonMatchs(1);//.withAddingCountOfPoints(MatchGain.W.getValue());
    					currentRankingAwayTeam=currentRankingAwayTeam.withAddingCountOfLostMatchs(1);//.withAddingCountOfPoints(MatchGain.L.getValue());
    				}
    				else if (matchEval.equals(MatchEvaluation.RESULT_2))
    				{
    					currentRankingHomeTeam=currentRankingHomeTeam.withAddingCountOfLostMatchs(1);//.withAddingCountOfPoints(MatchGain.L.getValue());    					
    					currentRankingAwayTeam=currentRankingAwayTeam.withAddingCountOfWonMatchs(1);//.withAddingCountOfPoints(MatchGain.W.getValue());
    				}
    				else if (matchEval.equals(MatchEvaluation.RESULT_N))
    				{
    					currentRankingHomeTeam=currentRankingHomeTeam.withAddingCountOfDrawMatchs(1);//.withAddingCountOfPoints(MatchGain.N.getValue());
    					currentRankingAwayTeam=currentRankingAwayTeam.withAddingCountOfDrawMatchs(1);//.withAddingCountOfPoints(MatchGain.N.getValue());

    				}
    				else throw new IllegalStateException("Unknown match evaluation! -> '"+matchEval+"'");
    				
    				/* save rankings in temporary map */
    				mapRankingByTeam.put(m.getHomeTeam(),currentRankingHomeTeam);
    				mapRankingByTeam.put(m.getAwayTeam(),currentRankingAwayTeam);    				
    			}
    		}
    		/*else
    		{
    			System.err.println("skipp matchweekorder="+mw.getMatchWeekOrder()+" because currentMatchWeekOrder is "+currentMatchWeekOrder);
    		}*/
    	}

    	/* consolidate results */    	    	    
    	final LeagueSeasonRanking computedRanking=new LeagueSeasonRanking(seasonYear,currentMatchWeekOrder);
    	for (LeagueSeasonRankingByTeam rlbt:mapRankingByTeam.values()) 
    	{	
    		final LeagueSeasonRankingByTeam consolidated=rlbt.withCountOfPoints(rlbt.getCountOfWonMatchs()*MatchGain.W.getValue()+rlbt.getCountOfDrawMatchs()*MatchGain.N.getValue());
    		computedRanking.addLeagueSeasonRankingByTeam(consolidated);
    	}
    	
    	return computedRanking;
    }
}
