/*
 * Copyright (C) 2010 Olivier PARISOT <parisot_olivier@yahoo.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.doopyon.ravanelab.datasource;

import java.util.*;
import org.doopyon.ravanelab.etl.domain.*;


/**
 * Abstraction of a league data provider.
 * 
 * @author Olivier PARISOT
 */
public abstract class ExternalLeagueDataPacketProvider implements Iterator<ExternalLeagueDataPacket> 
{
	//
	// Instance fields
	//
	
	/** The working league name. */
	private String leagueName;

	/** */
	private int endSeasonYear;
	/** */
	private int endMatchWeekOrder;
	
	/** The max count of match order weeks per year. */
	private int maxCountOfMatchOrderWeeksPerYear;
	
	/** The current season year. */
	private int currentSeasonYear;
	/** The current match week order. */
	private int currentMatchWeekOrder;	
	
	
	//
	// Instance methods
	//

	
	/**
	 * Get the league name.
	 * @return the leagueName
	 */
	public final String getLeagueName() 
	{
		return leagueName;
	}
	
	/**
	 * Initialize the data provider.
	 */
	public final void init(final String leagueName,final int startSeasonYear,final int startMatchWeekOrder,final int endSeasonYear,final int endMatchWeekOrder,final int maxMatchOrderWeekPerYear)
	{
		if (!acceptLeagueName(leagueName))
		{
			throw new IllegalArgumentException("Not initializable for the league '"+leagueName+"'!");
		}
		
		this.leagueName=leagueName;			
		
		this.endSeasonYear=endSeasonYear;		
		this.endMatchWeekOrder=endMatchWeekOrder;
		
		this.maxCountOfMatchOrderWeeksPerYear=maxMatchOrderWeekPerYear;
		
		this.currentSeasonYear=startSeasonYear;
		this.currentMatchWeekOrder=startMatchWeekOrder;		
	}
	
	
	/** 
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasNext() 
	{		
		return (currentSeasonYear<=endSeasonYear)&&(currentMatchWeekOrder<=endMatchWeekOrder);
	}

	/** 
	 * {@inheritDoc}
	 */
	@Override
	public ExternalLeagueDataPacket next() 
	{
		ExternalLeagueDataPacket eldp=null;
		try 
		{
			eldp=extractData(currentSeasonYear,currentMatchWeekOrder);
		} 
		catch (Exception e) 
		{
			e.printStackTrace();
		}
		
		currentMatchWeekOrder++;
		if (currentMatchWeekOrder>maxCountOfMatchOrderWeeksPerYear)
		{
			currentMatchWeekOrder=1;
			currentSeasonYear++;
		}
		
		return eldp;
	}		
	
	/**
	 * Extract data for the current league, a given season year and a given match week order.
	 * @param seasonYear a given season year
	 * @param matchWeekOrder a given match week order
	 * @return
	 */
	public abstract ExternalLeagueDataPacket extractData(final int seasonYear,final int matchWeekOrder) throws Exception; 

	/**
	 * Check if the current data provider is able to get data for a given league name.
	 * @param checkedLeagueName a given league name
	 * @return true if the current data provider is able to get data for a given league name, flase otherwise
	 */
	public abstract boolean acceptLeagueName(final String checkedLeagueName);

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void remove() 
	{
		throw new UnsupportedOperationException("not implemented method!");		
	}

}
