/*
 * Copyright (C) 2010 Olivier PARISOT <parisot_olivier@yahoo.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.doopyon.ravanelab.datasource;

import java.io.*;
import java.net.*;
import java.text.*;
import javax.swing.text.html.parser.*;
import org.apache.commons.logging.*;
import org.doopyon.ravanelab.etl.domain.*;
import org.doopyon.ravanelab.util.*;


/**
 * Data provider based on the website 'www.lfp.fr'.
 * 
 * @author Olivier PARISOT
 */
public final class LFPExternalLeagueDataPacketProvider extends ExternalLeagueDataPacketProvider
{
	//
	// Static fields
	//
	
	/** Logger. */
    private static final Log LOG=LogFactory.getLog(LFPExternalLeagueDataPacketProvider.class);		
	
	/** Commodity object used to parse hour format.*/
	private static final SimpleDateFormat HOUR_FORMAT=new SimpleDateFormat("H:m");
	/** Availables league names. */
	private static final String[] AVAILABLE_LEAGUES={"ligue1","ligue2"};
	
	/** Max count of tries to retrieve data. */
	private static final int MAX_TRIES_COUNT=3;
	
	
	//
	// Instance fields
	//

	/** */
	private final ParserDelegator parserDelegator;
	
	
	//
	// Constructor
	//
	
	/**
	 * Constructor.
	 */
	public LFPExternalLeagueDataPacketProvider()
	{
		super();
		this.parserDelegator=new ParserDelegator();		
	}

	
	//
	// Instance methods
	//
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalLeagueDataPacket extractData(final int seasonYear,final int matchWeekOrder) throws Exception
	{
		ExternalLeagueDataPacket eldp=null;
		for (int i=0;i<MAX_TRIES_COUNT;i++)
		{
			try
			{
				eldp=extractData0(seasonYear,matchWeekOrder);
			}
			catch(Exception e)
			{
				LOG.error("Error when extracting data form website! [try-count="+i+"/"+MAX_TRIES_COUNT+"]",e);
			}
			if (eldp!=null) break;
		}
		return eldp;
	}
	
	/**
	 * 
	 */
	public final ExternalLeagueDataPacket extractData0(final int seasonYear,final int matchWeekOrder) throws Exception
	{
		final String urlString=buildURLString(getLeagueName(),seasonYear,matchWeekOrder);
		final URL url=new URL(urlString);
		
		LOG.info("******* Parsing '"+urlString+"' ... *******");
		
		final URLConnection connection=url.openConnection();
		if (connection==null) throw new IOException("Unavailable connection");	
		
		final HTMLTableToCSVCallback parserCallback=new HTMLTableToCSVCallback();
		InputStream is=null;
		try
		{
			is=connection.getInputStream();
		}		
		catch(IOException e)
		{
			throw new Exception("Input stream not available!",e);
		}		
		final BufferedReader breader=new BufferedReader((new InputStreamReader(is)));
		try 
		{ 
			parserDelegator.parse(breader,parserCallback,true); 
		}
		catch(Exception e) 
		{ 
		  	e.printStackTrace(); 
		}        
		breader.close();
		
		final ExternalLeagueDataPacket eldp=new ExternalLeagueDataPacket(getLeagueName(),seasonYear,matchWeekOrder);
		for (String[] line:parserCallback.getTable())
		{
			/* check if line is processable */
			try
			{
				HOUR_FORMAT.parse(String.valueOf(line[0]));
			}
			catch(ParseException e) 
			{
				if (LOG.isDebugEnabled()) LOG.debug("******* Skipping the line -> "+StringUtil.concat(line)+" *******");
				continue;
			}							
			
			try
			{
				if (LOG.isDebugEnabled()) LOG.debug("******* Processing the line -> "+StringUtil.concat(line)+" *******");
				
				final String homeTeamName=line[1];
				final String awayTeamName=line[5];
				
				int homeTeamPoints=-1;
				int awayTeamPoints=-1;												
				final String[] scoreString=line[3].split("-");
				if (scoreString.length==2)
				{
					try
					{
						homeTeamPoints=Integer.valueOf(scoreString[0].trim());
					}
					//catch(NumberFormatException nfe) {}
					catch(Exception ee) {}
					try
					{
						awayTeamPoints=Integer.valueOf(scoreString[1].trim());
					}
					//catch(NumberFormatException nfe) {}
					catch(Exception ee) {}				
				}
				eldp.addExternalLeagueDataList(new ExternalLeagueData(homeTeamName,awayTeamName,homeTeamPoints,awayTeamPoints));
				if (LOG.isDebugEnabled()) LOG.debug("******* Loading match: "+homeTeamName+" vs "+awayTeamName+": "+homeTeamPoints+"-"+awayTeamPoints+" *******");
			}		
			catch(Exception e) 
			{				
				LOG.error("Error when processing the line -> "+StringUtil.concat(line),e);
			}
		}
		
		LOG.info("******* ... end parsing '"+urlString+"' *******");
		
		return eldp;
	}		
		
	/**
	 * {@inheritDoc}
	 */
	@Override
	public final boolean acceptLeagueName(final String checkedLeagueName)
	{
		for (String ln:AVAILABLE_LEAGUES) if (ln.equals(checkedLeagueName)) return true;
		return false;
	}
	
	
	//
	// Static methods
	//
	
	/**
	 * Build the URL.
	 */
	private static String buildURLString(final String leagueName,final int seasonYear,final int matchWeekOrder)
	{	
		final StringBuilder sb=new StringBuilder();
		sb.append("http://www.lfp.fr/").append(leagueName)
		  .append("/historique/resultat.asp?saison=").append(seasonYear-1).append("/").append(seasonYear)
		  .append("&code_jr_tr=J").append((matchWeekOrder>=10)?""+matchWeekOrder:"0"+matchWeekOrder);				
		return sb.toString();
	}
	

}
