/*
 * Copyright (C) 2010 Olivier PARISOT <parisot_olivier@yahoo.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.doopyon.ravanelab.etl.serviceimpl;

import java.util.*;
import org.apache.commons.logging.*;
import org.doopyon.ravanelab.core.domain.*;
import org.doopyon.ravanelab.etl.domain.*;
import org.springframework.stereotype.*;


/**
 * Implementation of ExternalLeagueDataPacketImporterService.
 * 
 * @author Olivier PARISOT
 */
@Service(ExternalLeagueDataPacketImporterServiceImpl.BEAN_ID)
public class ExternalLeagueDataPacketImporterServiceImpl extends ExternalLeagueDataPacketImporterServiceImplBase 
{
	//
	// Static fields
	//
	
	/** Logger. */
    private static final Log LOG=LogFactory.getLog(ExternalLeagueDataPacketImporterServiceImpl.class);	

    
    //
    // Instance methods
    //    

    /**
     * {@inheritDoc}
     */
    @Override
    public final void importExternalLeagueDataPacket(final ExternalLeagueDataPacket packet) 
    {
    	importExternalLeagueDataPacket0(packet,true,false);
    }       

    /**
     * {@inheritDoc}
     */    
	@Override
	public void importExternalLeagueDataPacketsList(final List<ExternalLeagueDataPacket> packets) 
	{
		final int packetsCount=packets.size();
		for (int i=0;i<packetsCount;i++)
		{
			if (packets.get(i)!=null) importExternalLeagueDataPacket0(packets.get(i),false,(i==packetsCount-1));
		}		
	}    
    
    /**
     *
     */
    public final void importExternalLeagueDataPacket0(final ExternalLeagueDataPacket packet,final boolean flushIfNeeded,final boolean forceFlush) 
    {
    	LOG.info("---- Retrieve or create if needed the '"+packet.getLeagueName()+"' league data ... ----");
    	League involvedLeague=getLeagueService().findLeagueByName(packet.getLeagueName());
    	if (involvedLeague==null)
    	{
    		involvedLeague=new League(packet.getLeagueName());
    		involvedLeague.addLeagueSeason(new LeagueSeason(packet.getSeasonYear()));
    		involvedLeague=getLeagueService().save(involvedLeague);
    	}
    	else
    	{
    		LOG.info("---- Retrieve or create if needed the '"+packet.getSeasonYear()+"' league season data ... ----");    	
    		if (involvedLeague.getLeagueSeasonForYear(packet.getSeasonYear())==null)
    		{
    			involvedLeague.addLeagueSeason(new LeagueSeason(packet.getSeasonYear()));
    			involvedLeague=getLeagueService().merge(involvedLeague);
    		}
    	}
    	final LeagueSeason involvedLeagueSeason=involvedLeague.getLeagueSeasonForYear(packet.getSeasonYear());
    	
    	LOG.info("---- Retrieve or create if needed the '"+packet.getMatchWeekOrder()+"' match week data ... ----");
    	MatchWeek involvedMatchWeek=involvedLeagueSeason.getMatchWeek(packet.getMatchWeekOrder());
    	if (involvedMatchWeek==null)
    	{
    		involvedMatchWeek=new MatchWeek(packet.getMatchWeekOrder());
    		involvedLeagueSeason.addMatchWeek(involvedMatchWeek);
    	}
    	    	
    	if (packet.getExternalLeagueDataList().size()>0)
    	{
    		LOG.info("---- Add or update the matchs ... ----");
	    	for (ExternalLeagueData d:packet.getExternalLeagueDataList())
	    	{
	    		final Match existingMatch=involvedMatchWeek.getMatchWithTeamsNames(d.getHomeTeamName(),d.getAwayTeamName());
	    		if (existingMatch!=null)
	    		{
	    			/* update match */
	    			if (d.getHomeTeamScore()>=0&&d.getAwayTeamScore()>=0)
	    			{
	    				existingMatch.setScore(new Score(d.getHomeTeamScore(),d.getAwayTeamScore()));
	    			}   	    			
	    		}
	    		else
	    		{
	    			/* create match */
	    			final Match newMatch=new Match();	    				
	    			newMatch.setHomeTeam(searchAndCreateIfNeeded(d.getHomeTeamName()));    		   	
	    			newMatch.setAwayTeam(searchAndCreateIfNeeded(d.getAwayTeamName()));
	     		
	    			if (d.getHomeTeamScore()>=0&&d.getAwayTeamScore()>=0)
	    			{
	    				newMatch.setScore(new Score(d.getHomeTeamScore(),d.getAwayTeamScore()));
	    			}    		
	    			involvedMatchWeek.addMatch(newMatch);
	    		}
	    	}
	    	
	    	if (flushIfNeeded)
	    	{
	    		LOG.info("---- Update the '"+packet.getLeagueName()+"' league data ... ----");
	    		getLeagueService().merge(involvedLeague);
	    	}
    	}
    	
    	if (forceFlush)
    	{
    		LOG.info("---- Update the '"+packet.getLeagueName()+"' league data ... ----");
    		getLeagueService().merge(involvedLeague);
    	}
    }        

    /**
     * Get, and create if needed, a team object corresponding to a team name.
     * @param ctx the service context
     * @param teamName the team name
     * @return the team object
     */
    private final Team searchAndCreateIfNeeded(final String teamName)
    {
    	final String teamNameWithoutQuote=teamName.replaceAll("'","");
    	Team involvedTeam=getTeamService().findTeamByName(teamNameWithoutQuote);
    	if (involvedTeam==null)
    	{
    		involvedTeam=getTeamService().save(new Team(teamNameWithoutQuote));
    	}
    	return involvedTeam;		
    }	
	
}
