/*
 * Copyright (C) 2010 Olivier PARISOT <parisot_olivier@yahoo.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.doopyon.ravanelab.helpers.serviceimpl;

import org.doopyon.ravanelab.core.domain.*;
import org.springframework.stereotype.*;
import java.util.*;


/**
 * Implementation of FakeLeagueSeasonManagerService.
 * 
 * @author Olivier PARISOT
 */
@Service(FakeLeagueSeasonManagerServiceImpl.BEAN_ID)
public class FakeLeagueSeasonManagerServiceImpl extends FakeLeagueSeasonManagerServiceImplBase 
{
	//
	// Static fields
	//
	
	/** */
	private static final int MAX_HOME_SCORE=5;
	/** */
	private static final int MAX_AWAY_SCORE=3;
	
	
	//
	// Instance methods
	//

    /**
     * {@inheritDoc}
     */	
	@Override
	public final LeagueSeason buildFakeLeagueSeason(final int seasonYear,final Set<Team> teams) 
	{   	
    	final int nbTeams=teams.size();
    	
    	if (nbTeams<=0) throw new IllegalArgumentException("Number of teams <= 0");
    	    	        	    	
    	final LeagueSeason ls=new LeagueSeason(seasonYear);

    	final int nbMatchWeek=2*(nbTeams-1);
    	final List<Match> allPossibleMatchs=buildAllPossiblesMatchs(teams);
    	for (int i=0;i<nbMatchWeek;i++)
    	{
    		final Set<Team> alreadyInCurrentMatchWeek=new HashSet<Team>();
    		
    		final MatchWeek mw=new MatchWeek(i+1);
    		
    		for (Iterator<Match> iter=allPossibleMatchs.iterator();iter.hasNext()&&mw.getMatchs().size()<nbTeams/2;)
    		{
    			final Match suggestedMatch=iter.next();
    			final Team hTeam=suggestedMatch.getHomeTeam();
    			final Team aTeam=suggestedMatch.getAwayTeam();
    			if (!alreadyInCurrentMatchWeek.contains(hTeam)&&!alreadyInCurrentMatchWeek.contains(aTeam))
    			{
    				mw.addMatch(suggestedMatch);
    				alreadyInCurrentMatchWeek.add(hTeam);
    				alreadyInCurrentMatchWeek.add(aTeam);
    				iter.remove();
    			}    			    			
    		}
    		
    		ls.addMatchWeek(mw);
    	}
    	
        return ls;
	}

    /**
     * {@inheritDoc}
     */	
	@Override
	public final void simulateLeagueSeason(final LeagueSeason leagueSeason,final int lastWeekOrderToSimulate) 
	{
		if (leagueSeason==null) throw new IllegalArgumentException("leagueSeason is null!");
		if (lastWeekOrderToSimulate<=0) throw new IllegalArgumentException("lastWeekOrderToSimulate <= 0");
		
		for (int currentWeekOrder=1;currentWeekOrder<=lastWeekOrderToSimulate;currentWeekOrder++)
		{
			final MatchWeek currentMatchWeek=leagueSeason.getMatchWeek(currentWeekOrder);
			for (Match m:currentMatchWeek.getMatchs())
			{
				if (m.getScore()!=null) throw new IllegalStateException("Match ["+m+"] already played!");				
				m.setScore(new Score((int)Math.round(Math.random()%MAX_HOME_SCORE),
									 (int)Math.round(Math.random()%MAX_AWAY_SCORE)));
			}
		}		
	}
	
	
    //
    // Static methods
    //
    
    /**
     * Build a list of all possibles matchs for a given set of Teams.
     * @return the list of all possibles matchs
     */
    public static final List<Match> buildAllPossiblesMatchs(final Set<Team> teams)
    {
    	final List<Match> allPossibleMatchs=new ArrayList<Match>();
    	
    	for (Team homeTeam:teams)
    	{
    		for (Team awayTeam:teams)
    		{
    			if (!homeTeam.getTeamName().equals(awayTeam.getTeamName())) // TODO: bof bof... 
    			{
    				final Match match=new Match();
    				match.setHomeTeam(homeTeam);
    				match.setAwayTeam(awayTeam);
    				allPossibleMatchs.add(match);
    			}
    		}
    	}
    	
    	return allPossibleMatchs;
    }	
}
