/*
 * Copyright (C) 2010 Olivier PARISOT <parisot_olivier@yahoo.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.doopyon.ravanelab.lab.serviceimpl;

import java.util.*;
import org.doopyon.ravanelab.core.domain.*;
import org.doopyon.ravanelab.core.serviceapi.*;
import org.doopyon.ravanelab.lab.serviceapi.*;
import org.doopyon.ravanelab.util.*;
import org.springframework.beans.factory.annotation.*;
import org.springframework.stereotype.Service;


/**
 *
 *
 * @author Olivier PARISOT
 *
 */
@Service(ClassifierService.BEAN_ID)
public class ClassifierServiceImpl implements ClassifierService 
{
	//
	// Instance fields
	//
	
	/** */
	private LeagueSeasonRankingService leagueSeasonRankingService;
	/** */
	private LogicMatchEvaluationComputerService computer;
	/** */
	private MatchEvaluatorService evaluator;	
	
	//
	// Instance methods
	//
	
	/**
	 * 
	 * @param leagueSeasonRankingService the leagueSeasonRankingService to set
	 */
	@Autowired
	public void setLeagueSeasonRankingService(final LeagueSeasonRankingService leagueSeasonRankingService) 
	{
		this.leagueSeasonRankingService=leagueSeasonRankingService;
	}	
	
	/**
	 * 
	 * @param computer the computer to set
	 */
	@Autowired
	public void setComputer(final LogicMatchEvaluationComputerService computer) 
	{
		this.computer=computer;
	}

	/**
	 * 
	 * @param evaluator the evaluator to set
	 */
	@Autowired
	public void setEvaluator(final MatchEvaluatorService evaluator) 
	{
		this.evaluator=evaluator;
	}	
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Repartition<MatchEvaluationClassification> getLeagueSeasonClassificationsRepartition(final League l,final int seasonYear) 
	{
		final Repartition<MatchEvaluationClassification> r=new Repartition<MatchEvaluationClassification>();
		final LeagueSeason ls=l.getLeagueSeasonForYear(seasonYear);
		for (MatchWeek mw:ls.getMatchWeeks())
		{
			final LeagueSeasonRanking lsr=leagueSeasonRankingService.getLeagueSeasonRanking(l,seasonYear,mw.getMatchWeekOrder()-1);
			for (Match m:mw.getMatchs())
			{
				r.addOccurenceFor(getMatchClassification(m,lsr));				
			}
		}
		return r;
	}

	/**
	 * {@inheritDoc} 
	 */
	@Override
	public final MatchEvaluationClassification getMatchClassification(final Match m,final LeagueSeasonRanking lsr) 
	{	
		final MatchEvaluation eval=evaluator.getMatchEvaluation(m);
		if (eval.equals(MatchEvaluation.NOTPLAYED)) throw new IllegalStateException("Impossible to classify a not played match!");
		
		final boolean isLogic=computer.getComputedLogicMatchEvaluations(m,lsr).contains(eval);
		final boolean isNotLogic=computer.getComputedNotLogicMatchEvaluations(m,lsr).contains(eval);
		
		if (isLogic&&isNotLogic) throw new IllegalStateException("{match="+m+"result="+eval+" is logic and not logic in the same time?");
		
		if (isLogic) return MatchEvaluationClassification.LOGIC;
		else if (isNotLogic) return MatchEvaluationClassification.NOT_LOGIC;
		else return MatchEvaluationClassification.UNDEFINED;
	}

	/**
	 * {@inheritDoc} 
	 */
	@Override
	public Map<Match,MatchEvaluationClassification> getMatchWeekClassifications(final League l,final int seasonYear,final int matchWeekOrder) 
	{
		final Map<Match,MatchEvaluationClassification> r=new HashMap<Match,MatchEvaluationClassification>();
		final LeagueSeason ls=l.getLeagueSeasonForYear(seasonYear);
		final MatchWeek mw=ls.getMatchWeek(matchWeekOrder);		
		final LeagueSeasonRanking lsr=leagueSeasonRankingService.getLeagueSeasonRanking(l,seasonYear,mw.getMatchWeekOrder()-1);
		for (Match m:mw.getMatchs()) r.put(m,getMatchClassification(m,lsr));						
		return r;
	}


}
