/*
 * Copyright (C) 2010 Olivier PARISOT <parisot_olivier@yahoo.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.doopyon.ravanelab.launcher;

import java.util.*;
import org.apache.commons.cli.*;
import org.doopyon.ravanelab.core.domain.*;
import org.doopyon.ravanelab.core.serviceapi.*;
import org.doopyon.ravanelab.datasource.*;
import org.doopyon.ravanelab.etl.serviceapi.*;
import org.doopyon.ravanelab.lab.serviceapi.*;
import org.doopyon.ravanelab.ext.serviceapi.*;
import org.doopyon.ravanelab.prediction.domain.*;
import org.doopyon.ravanelab.prediction.serviceapi.*;
import org.springframework.context.*;
import org.springframework.context.support.*;


/**
 * Command line interface.
 * 
 * @author Olivier PARISOT
 */
public final class Main 
{
	//
	// Static fields
	//

	/** */
	private static final int SEASON_IN_PROGRESS=Calendar.getInstance().get(Calendar.YEAR)+1;
	/** */
	private static final int MIN_SEASON_TO_KEEP_IN_ACCOUNT=2002;
	
	/** Command identifier: fillDB. */
	private static final String COMMAND_FILLDB="fillDB";
	/** Command identifier: predict. */
	private static final String COMMAND_PREDICT="predict";
	/** Command identifier: ranking. */
	private static final String COMMAND_RANKING="ranking";
	/** Command identifier: exportARFF. */
	private static final String COMMAND_EXPORTARFF="exportARFF";
	/** Command identifier: classify. */
	private static final String COMMAND_CLASSIFY="classify";
	
	/** Argument identifier: leagueName. */
	private static final String ARGUMENT_LEAGUENAME="leagueName";
	/** Argument identifier: seasonYear. */
	private static final String ARGUMENT_SEASONYEAR="seasonYear";
	/** Argument identifier: matchWeekOrder. */
	private static final String ARGUMENT_MATCHWEEKORDER="matchWeekOrder";
	
	
	//
	// Constructor
	//
	
	/**
	 * Private constructor to avoid instanciation.
	 */
	private Main() {}
	
	
	//
	// Main method
	//	
	
	/**
	 * Main method.
	 * @param args command line arguments
	 */
	public static void main(String[] args) 
	{
		/* init application context */
		final AbstractApplicationContext context=new ClassPathXmlApplicationContext(new String[]{"classpath:/applicationContext.xml"});
		
		/* parse command line and run related commands */
	    final CommandLineParser parser=new GnuParser();	    
	    final Options options=new Options();	    	    
	    for (Option opt:createCommandLineOptionsList()) options.addOption(opt);	    
	    System.out.println("\n");
	    
	    try 
	    {	    	
	        final CommandLine line=parser.parse(options,args);	        
	        if (line.hasOption(COMMAND_PREDICT))
	        {
	        	predict(context,line.getOptionValue(ARGUMENT_LEAGUENAME),
	        					Integer.parseInt(line.getOptionValue(ARGUMENT_SEASONYEAR,"-1")),
	        					Integer.parseInt(line.getOptionValue(ARGUMENT_MATCHWEEKORDER,"-1")));
	        }
	        else if (line.hasOption(COMMAND_RANKING))
	        {
	        	ranking(context,line.getOptionValue(ARGUMENT_LEAGUENAME),
	        					Integer.parseInt(line.getOptionValue(ARGUMENT_SEASONYEAR,"-1")),
	        					Integer.parseInt(line.getOptionValue(ARGUMENT_MATCHWEEKORDER,"-1")));
	        }
	        else if (line.hasOption(COMMAND_FILLDB))
	        {
	        	fillDB(context,line.getOptionValue(ARGUMENT_LEAGUENAME),
	        				   Integer.parseInt(line.getOptionValue(ARGUMENT_SEASONYEAR,"-1")));
	        }
	        else if (line.hasOption(COMMAND_EXPORTARFF))
	        {
	        	exportARFF(context,line.getOptionValue(ARGUMENT_LEAGUENAME),MIN_SEASON_TO_KEEP_IN_ACCOUNT,1,SEASON_IN_PROGRESS,38);    	
	        }
	        else if (line.hasOption(COMMAND_CLASSIFY))
	        {
	        	classify(context,line.getOptionValue(ARGUMENT_LEAGUENAME),
	        					 Integer.parseInt(line.getOptionValue(ARGUMENT_SEASONYEAR,"-1")),
	        					 Integer.parseInt(line.getOptionValue(ARGUMENT_MATCHWEEKORDER,"-1")));    	
	        }		        
	        else
	        {
	        	printHelp(options);
	        }
	    }
	    catch(Exception e) 
	    {
	        System.err.println("Command line parsing failed: "+e.getMessage());
	        e.printStackTrace();
	        printHelp(options);
	    }

		
		/* end */
	    context.registerShutdownHook();
		System.exit(0);
	}

	/**
	 * Command implementation: classify.
	 */	
	private static void classify(final ApplicationContext context,final String leagueName,final int seasonYear,final int matchWeekOrder) 
	{
		/* check arguments */
		if (leagueName==null) throw new IllegalArgumentException("Missing league name!");
		if (seasonYear<0) throw new IllegalArgumentException("Missing season year!");
		
		/* compute and print classifications */
		final LeagueService ls=(LeagueService)context.getBean(LeagueService.BEAN_ID);
		final League l=ls.findLeagueByName(leagueName);
		if (l==null) throw new IllegalArgumentException("Unknown league!");		
		final ClassifierService lscs=(ClassifierService)context.getBean(ClassifierService.BEAN_ID);		
		if (matchWeekOrder==-1)
		{
			System.out.println(lscs.getLeagueSeasonClassificationsRepartition(l,seasonYear).toString());
		}
		else
		{
			System.out.println(lscs.getMatchWeekClassifications(l,seasonYear,matchWeekOrder).toString());
		}
	}


	/**
	 * Command implementation: ranking.
	 */
	private static void ranking(final ApplicationContext context,final String leagueName,final int seasonYear,final int matchWeekOrder) 
	{
		/* check arguments */
		if (leagueName==null) throw new IllegalArgumentException("Missing league name!");
		if (seasonYear<0) throw new IllegalArgumentException("Missing season year!");
		if (matchWeekOrder<0) throw new IllegalArgumentException("Missing match week order!");
		
		/* compute ranking */
		final LeagueService ls=(LeagueService)context.getBean(LeagueService.BEAN_ID);
		final League l=ls.findLeagueByName(leagueName);
		if (l==null) throw new IllegalArgumentException("Unknown league!");
		final LeagueSeasonRankingService rankingService=(LeagueSeasonRankingService)context.getBean(LeagueSeasonRankingService.BEAN_ID);
		final LeagueSeasonRanking ranking=rankingService.getLeagueSeasonRanking(l,seasonYear,matchWeekOrder);
		
		/* print results! */
		System.out.println(l.getLeagueSeasonForYear(seasonYear));
		System.out.println(ranking.toString());	
	}


	/**
	 * Command implementation: fillDB.
	 */
	@SuppressWarnings("static-access")
	private static void fillDB(final ApplicationContext context,final String leagueName,final int seasonYear) 
	{
		/* check arguments */		
		if (leagueName==null) throw new IllegalArgumentException("Missing league name!");
			
		/* get the data provider */
		final ExternalLeagueDataPacketImporterService importer=(ExternalLeagueDataPacketImporterService)context.getBean(ExternalLeagueDataPacketImporterService.BEAN_ID);		
		final ExternalLeagueDataPacketProviderFactory providersFactory=(ExternalLeagueDataPacketProviderFactory)context.getBean(ExternalLeagueDataPacketProviderFactory.BEAN_ID);
		final ExternalLeagueDataPacketProvider provider=providersFactory.getProvider(leagueName);
		final ExternalLeagueDataPacketsListProvider listProvider=new SimpleExternalLeagueDataPacketsListProvider(provider,10);
			
		/* use the data provider and import data */
		if (seasonYear==-1)
		{
			for (int year=SEASON_IN_PROGRESS;year>MIN_SEASON_TO_KEEP_IN_ACCOUNT;year--)
			{
				provider.init(leagueName,year,1,year,38,38);
				while (listProvider.hasNext()) importer.importExternalLeagueDataPacketsList(listProvider.next());
			}			
		}
		else
		{
			provider.init(leagueName,seasonYear,1,seasonYear,38,38);
			while (listProvider.hasNext()) importer.importExternalLeagueDataPacketsList(listProvider.next());
		}
		
		/* sleep */
		try 
		{
			Thread.currentThread().sleep(2000);
		} 
		catch (InterruptedException e) {}
	}

	/**
	 * Create the list of command line options.
	 */
	@SuppressWarnings("static-access")
	private static List<Option> createCommandLineOptionsList()
	{
		final List<Option> l=new ArrayList<Option>();
		l.add(OptionBuilder.withDescription("predict results").create(COMMAND_PREDICT));
		l.add(OptionBuilder.withDescription("ranking").create(COMMAND_RANKING));
		l.add(OptionBuilder.withDescription("fill the local DB").create(COMMAND_FILLDB));
		l.add(OptionBuilder.withDescription("export data into ARFF format").create(COMMAND_EXPORTARFF));
		l.add(OptionBuilder.withDescription("classify").create(COMMAND_CLASSIFY));
		l.add(OptionBuilder.withArgName("leagueName").hasArg().withDescription("league name").create(ARGUMENT_LEAGUENAME));
		l.add(OptionBuilder.withArgName("seasonYear").hasArg().withDescription("season year").create(ARGUMENT_SEASONYEAR));
		l.add(OptionBuilder.withArgName("matchWeekOrder").hasArg().withDescription("match week order").create(ARGUMENT_MATCHWEEKORDER));
		return l;		
	}	

	/**
	 * Print help.
	 */
	private static void printHelp(Options options)
	{
    	final HelpFormatter formatter=new HelpFormatter();
    	formatter.printHelp("java -jar XXXX.jar",options);		
	}	
	
	/**
	 * Command implementation: predict.
	 * @param context
	 * @param leagueName
	 * @param seasonYear
	 * @param matchWeekOrder
	 */
	private static void predict(final ApplicationContext context,final String leagueName,final int seasonYear,final int matchWeekOrder)
	{
		/* check arguments */
		if (leagueName==null) throw new IllegalArgumentException("Missing league name!");
		if (seasonYear<0) throw new IllegalArgumentException("Missing season year!");
		if (matchWeekOrder<0) throw new IllegalArgumentException("Missing match week order!");
		
		/* compute predictions */
		final MatchPredictionFacadeService ls=(MatchPredictionFacadeService)context.getBean(MatchPredictionFacadeService.BEAN_ID);		
		final List<MatchPrediction> r=ls.getMatchWeekPrediction(leagueName,seasonYear,matchWeekOrder);
		
		/* print predictions */
		System.out.println("Predictions for league="+leagueName+" seasonYear="+seasonYear+" matchWeekOrder="+matchWeekOrder+":");
		for (MatchPrediction mp:r)
		{
			System.out.println(mp.getHomeTeam()+"-"+mp.getAwayTeam()+"\t -> \t"+mp.getPredictedEvaluation());
		}		
	}

	/**
	 * 
	 * @param context
	 * @param leagueName
	 * @param startSeasonYear
	 * @param startMatchWeekOrder
	 * @param endSeasonYear
	 * @param endMatchWeekOrder
	 */
	private static final void exportARFF(final ApplicationContext context,final String leagueName,int startSeasonYear,int startMatchWeekOrder,int endSeasonYear,int endMatchWeekOrder)
	{
		/* check arguments */
		if (leagueName==null) throw new IllegalArgumentException("Missing league name!");

		/* init services */
		final LeagueService leagueService=(LeagueService)context.getBean(LeagueService.BEAN_ID);
		final League league=leagueService.findLeagueByName(leagueName);
		if (league==null) throw new IllegalArgumentException("Unknown league!");
		final LeagueSeasonRankingService leagueSeasonRankingService=(LeagueSeasonRankingService)context.getBean(LeagueSeasonRankingService.BEAN_ID);		
		final MatchEvaluatorService matchEvaluatorService=(MatchEvaluatorService)context.getBean(MatchEvaluatorService.BEAN_ID);
		final TeamInfosFinderService teamInfosFinderService=(TeamInfosFinderService)context.getBean(TeamInfosFinderService.BEAN_ID);
		
		/* init results */
		final StringBuilder sb=new StringBuilder();
		sb.append("@relation stats_").append(leagueName).append("\n\n");		
		sb.append("@attribute totalCountOfPlayedMatchs numeric \n");
		sb.append("@attribute homeTeamRankByPoints numeric \n");
		sb.append("@attribute awayTeamRankByPoints numeric \n");		
		sb.append("@attribute homeTeamRankByGoalsFor numeric \n");
		sb.append("@attribute awayTeamRankByGoalsFor numeric \n");
		sb.append("@attribute homeTeamRankByGoalsAgainst numeric \n");
		sb.append("@attribute awayTeamRankByGoalsAgainst numeric \n");
		sb.append("@attribute homeTeamBudget numeric \n");
		sb.append("@attribute awayTeamBudget numeric \n");
		sb.append("@attribute result {RESULT_1,RESULT_N,RESULT_2,UNDEFINED} \n");
		sb.append("@data\n\n");
		
		/* build results */
		int cnt=0;	
		for (int currentYear=startSeasonYear;currentYear<=endSeasonYear;currentYear++)
		{	
			final LeagueSeason ls=league.getLeagueSeasonForYear(currentYear);
			if (ls==null) continue;				
			for (MatchWeek mw:ls.getMatchWeeks())
			{
				if (currentYear==startSeasonYear&&mw.getMatchWeekOrder()<startMatchWeekOrder) continue;
				if (currentYear==endSeasonYear&&mw.getMatchWeekOrder()>endMatchWeekOrder) continue;				

				final int currentMatchWeekOrder=mw.getMatchWeekOrder()-1;				
				
				final LeagueSeasonRanking lsr=leagueSeasonRankingService.getLeagueSeasonRanking(league,currentYear,currentMatchWeekOrder);				
				for (Match m:mw.getMatchs())
				{		
					try
					{
						final Team homeTeam=m.getHomeTeam();
						final Team awayTeam=m.getAwayTeam();
						final LeagueSeasonRankingByTeam homeLeagueSeasonRanking=lsr.getLeagueSeasonRankingByTeam(homeTeam);
						//final LeagueSeasonRankingByTeam awayLeagueSeasonRanking=lsr.getLeagueSeasonRankingByTeam(awayTeam);
						
						final int totalCountOfPlayedMatchs=homeLeagueSeasonRanking.getTotalCountOfPlayedMatchs();//,nbMatchWeeks);
						final int homeTeamRankByPoints=lsr.getRankOfTeamByPoints(homeTeam);//,nbTeams);
						final int awayTeamRankByPoints=lsr.getRankOfTeamByPoints(awayTeam);//,nbTeams);
						final int homeTeamRankByGoalsFor=lsr.getRankOfTeamByGoalsFor(homeTeam);//,nbTeams);
						final int awayTeamRankByGoalsFor=lsr.getRankOfTeamByGoalsFor(awayTeam);//,nbTeams);
						final int homeTeamRankByGoalsAgainst=lsr.getRankOfTeamByGoalsAgainst(homeTeam);//,nbTeams);
						final int awayTeamRankByGoalsAgainst=lsr.getRankOfTeamByGoalsAgainst(awayTeam);//,nbTeams);
						
						final MatchEvaluation matchEval=matchEvaluatorService.getMatchEvaluation(m);
						if (matchEval.equals(MatchEvaluation.NOTPLAYED)) continue;
						
						sb
						  .append(totalCountOfPlayedMatchs).append(',')
						  .append(homeTeamRankByPoints).append(',')
						  .append(awayTeamRankByPoints).append(',')
						  .append(homeTeamRankByGoalsFor).append(',')
						  .append(awayTeamRankByGoalsFor).append(',')
						  .append(homeTeamRankByGoalsAgainst).append(',')
						  .append(awayTeamRankByGoalsAgainst).append(',')
						  .append(teamInfosFinderService.findBudgetByTeamAndSeasonYear(homeTeam,currentYear)).append(',')
						  .append(teamInfosFinderService.findBudgetByTeamAndSeasonYear(awayTeam,currentYear)).append(',')	
						  .append(matchEval.getName()).append('\n');											
						cnt++;
					}
					catch(Exception e)
					{
						throw new IllegalStateException("Impossible to encode [l='"+league.getLeagueName()+"',ls='"+currentYear+"',mwo='"+mw.getMatchWeekOrder()+"',match='"+m+"']",e);
					}
				}
			}
		}

		/* print results */
		System.out.println(sb.toString());
	}
}
