/*
 * Copyright (C) 2010 Olivier PARISOT <parisot_olivier@yahoo.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */package org.doopyon.ravanelab.nn.domain;

import java.util.*;

import org.doopyon.ravanelab.util.MathsUtil;


/**
 * Value object representing NeuralNetworkData.
 * This class is responsible for the domain object related
 * business logic for NeuralNetworkData. Properties and associations are
 * implemented in the generated base class {@link org.doopyon.ravanelab.nn.domain.NeuralNetworkDataBase}.
 * 
 * To be used as input and output.
 * Can only contain double (0<d<1).
 * 
 * @author Olivier PARISOT
 */
public class NeuralNetworkData extends NeuralNetworkDataBase 
{
	//
	// Static fields
	//
	
	/** Serial version UID. */
	private static final long serialVersionUID=7774680637050041312L;

	
	//
	// Constructors
	//	

	/**
	 * Constructor.
	 */
    protected NeuralNetworkData() {}

    /**
     * Constructor.
     * @param size
     */
    public NeuralNetworkData(final int size) 
    {
        this(new ArrayList<Double>(size));
        final List<Double> l=getData();
        for (int i=0;i<size;i++) l.add(0d);
    }      
    
    /**
     * Constructor.
     * @param data
     */
    public NeuralNetworkData(final List<Double> data) 
    {
        super(checkValues(data));
    }   

    /**
     * Constructor.
     * @param vals
     */
    public NeuralNetworkData(double... vals) 
    {
        this(doublesAsList(vals));
    }      
    
    
    //
    // Instance methods
    //
    
    /**
     * Get the values into an array of doubles.
     */
    public final double[] getDoublesArray()
    {
        final List<Double> l=getData();
        final int lsize=l.size();
    	final double[] d=new double[lsize];
    	for (int i=0;i<lsize;i++) d[i]=l.get(i).doubleValue();
    	return d;
    }
    
    /**
     * Get the size.
     * @return the size
     */
    public final int size()
    {
    	return getData().size();
    }
    
    /**
     * 
     * @param i
     * @return
     */
    public final double get(final int idx)
    {
    	return getData().get(idx);
    }
    
    /**
     * 
     * @param i
     * @return
     */
    public final double set(final int idx,final double value)
    {
		checkValue(value);
    	return getData().set(idx,value);
    }

    /**
     * 
     * @param idx
     */
    public final NeuralNetworkData withValueEqualsToOneAt(final int idx)
    {
    	set(idx,1d);
    	return this;
    }
    
    /**
     * 
     * @param neuralNetworkData
     */
	public final void add(final NeuralNetworkData neuralNetworkData) 
	{
		add(neuralNetworkData.getData());		
	}

    /**
     * 
     * @param neuralNetworkData
     */
	public final void add(final List<Double> l) 
	{
		getData().addAll(l);		
	}	
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String toString()
	{
		return getData().toString();
	}
	
	/**
	 * 
	 */
	public final int getIndexOfFirstMaxValue()
	{
		return MathsUtil.getIndexOfMaxValue(getDoublesArray());
	}
    
	/**
	 * 
	 * @return
	 */
	/*public final double getMaxValue0()
	{
		double maxVal=Double.MIN_VALUE;
		for (Double d:getData()) if (d.doubleValue()>maxVal) maxVal=d.doubleValue();
		return maxVal;
	}*/
	
	
    //
    // Static methods
    //
    
    /**
     * Commodity method to transform a variable serie of doubles into a list of doubles.
     */
    private static List<Double> doublesAsList(final double... vals)
    {
    	final List<Double> l=new ArrayList<Double>();
    	for (double d:vals) l.add(Double.valueOf(d));
    	return l;
    }

    /**
     * Commodity method used to check range of a list of doubles (should be between 0 and 1).
     * @param listOfDoubles the list of doubles
     */
    private static List<Double> checkValues(final List<Double> listOfDoubles)
    {
    	for (Double d:listOfDoubles)
    	{
    		checkValue(d.doubleValue());
    	}
    	return listOfDoubles;
    }
    
    /**
     * Commodity method used to check range of a double (should be between 0 and 1).
     */    
    private static void checkValue(final double val)
    {
		if (val<0d) throw new IllegalArgumentException("Value can not be <0 in NeuralNetworkData!");
		if (val>1d) throw new IllegalArgumentException("Value can not be >1 in NeuralNetworkData!");    	
    }


}
