/*
 * Copyright (C) 2010 Olivier PARISOT <parisot_olivier@yahoo.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.doopyon.ravanelab.nn.serviceimpl;

import java.util.*;
import org.apache.commons.logging.*;
import org.doopyon.ravanelab.nn.domain.*;
import org.doopyon.ravanelab.nn.domain.util.*;
import org.encog.neural.activation.*;
import org.encog.neural.networks.*;
import org.encog.neural.networks.layers.*;
import org.springframework.stereotype.*;


/**
 * Implementation of NeuralNetworkBuilderService.
 * 
 * @author Olivier PARISOT
 */
@Service(NeuralNetworkBuilderServiceImpl.BEAN_ID)
public class NeuralNetworkBuilderServiceImpl extends NeuralNetworkBuilderServiceImplBase 
{
	//
	// Static fields
	//
	
	/** Logger. */
    private static final Log LOG=LogFactory.getLog(NeuralNetworkBuilderServiceImpl.class);

    
    //
    // Instance methods
    //
    
    /**
     * {@inheritDoc}
     */ 
 	@Override
	public final NeuralNetwork getNeuralNetwork(final int nbInput,final int nbOutput,final int nbNeuronsInHiddenLayer1,int nbNeuronsInHiddenLayer2) 
 	{
		final BasicNetwork internalNetwork=new BasicNetwork();
		internalNetwork.addLayer(new BasicLayer(new ActivationSigmoid(),true,nbInput));
		if (nbNeuronsInHiddenLayer1>0)
		{
			internalNetwork.addLayer(new BasicLayer(new ActivationSigmoid(),true,nbNeuronsInHiddenLayer1));
		}
		if (nbNeuronsInHiddenLayer2>0)
		{
			internalNetwork.addLayer(new BasicLayer(new ActivationSigmoid(),true,nbNeuronsInHiddenLayer2));
		}
		internalNetwork.addLayer(new BasicLayer(new ActivationSigmoid(),true,nbOutput));		
		internalNetwork.getStructure().finalizeStructure();		
		internalNetwork.reset();
		
		final StringBuilder sb=new StringBuilder();
		sb.append("{in=").append(nbInput);
		if (nbNeuronsInHiddenLayer1>0) sb.append(",h1=").append(nbNeuronsInHiddenLayer1);
		if (nbNeuronsInHiddenLayer2>0) sb.append(",h2=").append(nbNeuronsInHiddenLayer2);
		sb.append(",out=").append(nbOutput).append('}');		
		internalNetwork.setDescription(sb.toString());
 	
 		final NeuralNetwork nn=new NeuralNetwork();
		nn.setInternalEncogNeuralNetwork(internalNetwork);
		nn.setInputNeuronsCount(nbInput);
		nn.setOutputNeuronsCount(nbOutput);
		nn.setHiddenNeuronsCount(nbNeuronsInHiddenLayer1+nbNeuronsInHiddenLayer2);			
		
		return nn;
	}

    /**
     * {@inheritDoc}
     */ 	
	@Override
	public final List<NeuralNetwork> getNeuralNetworksList(final int nbInput,final int nbOutput,final NeuralNetworkBuilderConfiguration cfg) 
	{			
		final int minNeurons=nbInput/2;
		final int maxNeurons=2*nbInput;   // could be 3*inInput, according to the litterature
		
		final List<int[]> nnIdx=new ArrayList<int[]>();
		for (int i=maxNeurons;i>=minNeurons;i--)
		{
			if (!cfg.equals(NeuralNetworkBuilderConfiguration.ONLY_TWO_LAYERS))
			{
				nnIdx.add(new int[]{i,0});
			}			
			if (!cfg.equals(NeuralNetworkBuilderConfiguration.ONLY_ONE_LAYER))	
			{	
				for (int j=i-1;j>=minNeurons;j--) nnIdx.add(new int[]{i,j});
			}
		}		

		final int nnIdxSize=nnIdx.size();		
		if (nnIdxSize==0) throw new IllegalStateException("null count of neural networks to build?");		
		
		LOG.info("building "+nnIdxSize+" neural networks ...");		
		final List<NeuralNetwork> networksList=new ArrayList<NeuralNetwork>(nnIdxSize);		
		for (int i=0;i<nnIdxSize;i++)
		{
			final int[] t=nnIdx.get(i);
			networksList.add(getNeuralNetwork(nbInput,nbOutput,t[0],t[1]));
			if (i%10==0) LOG.info("... "+i+"/"+nnIdxSize+" neural networks built ...");
		}
		Collections.sort(networksList,new NeuralNetworkComparator());
		LOG.info("... "+nnIdxSize+" neural networks built");
		
		return networksList;
	}
}
