/*
 * Copyright (C) 2010 Olivier PARISOT <parisot_olivier@yahoo.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.doopyon.ravanelab.prediction.serviceimpl;

import java.util.*;
import org.apache.commons.logging.*;
import org.doopyon.ravanelab.codec.serviceapi.*;
import org.doopyon.ravanelab.core.domain.*;
import org.doopyon.ravanelab.core.serviceapi.*;
import org.doopyon.ravanelab.nn.domain.*;
import org.doopyon.ravanelab.prediction.domain.*;
import org.springframework.stereotype.*;


/**
 * Implementation of MatchPredictionContextBuilderService.
 * 
 * @author Olivier PARISOT
 */
@Service(MatchPredictionContextBuilderServiceImpl.BEAN_ID)
public class MatchPredictionContextBuilderServiceImpl extends MatchPredictionContextBuilderServiceImplBase 
{
	//
	// Static fields
	//
	
	/** Logger. */
    private static final Log LOG=LogFactory.getLog(MatchPredictionContextBuilderServiceImpl.class);	

    
	//
	// Instance methods
	//
	
	/**
	 * {@inheritDoc}	
	 */
	@Override
    public final MatchPredictionContext getMatchPredictionContext(final League league,int startSeasonYear,int startMatchWeekOrder,final int endSeasonYear,final int endMatchWeekOrder) 
	{
		/* check arguments */
		if (league==null) throw new IllegalArgumentException("league object is null?");
		
		/* init training data set */
		LOG.info("start ...");
		final List<NeuralNetworkData> inputs=new ArrayList<NeuralNetworkData>();
		final List<NeuralNetworkData> expectedOutputs=new ArrayList<NeuralNetworkData>();
		
		/* init services */
		final LeagueSeasonRankingService leagueSeasonRankingService=getLeagueSeasonRankingService();
		final MatchCodecService matchCodecService=getMatchCodecService();
		final MatchEvaluationCodecService matchEvaluationCodecService=getMatchEvaluationCodecService();
		final MatchEvaluatorService matchEvaluatorService=getMatchEvaluatorService();
		
		/* build training data set */
		LOG.info("build training data set ...");		
		for (int currentYear=startSeasonYear;currentYear<=endSeasonYear;currentYear++)
		{	
			final LeagueSeason ls=league.getLeagueSeasonForYear(currentYear);
			if (ls==null)
			{
				LOG.error("... skipp data, no season '"+currentYear+"' for league '"+league.getLeagueName()+"'");
				continue;
			}
			if (!ls.isComplete())
			{
				LOG.warn("... season '"+currentYear+"' for league '"+league.getLeagueName()+"' not complete!");
			}
			LOG.info("... build training data set for season '"+currentYear+"' ...");						
			for (MatchWeek mw:ls.getMatchWeeks())
			{
				if (currentYear==startSeasonYear&&mw.getMatchWeekOrder()<startMatchWeekOrder) continue;
				if (currentYear==endSeasonYear&&mw.getMatchWeekOrder()>endMatchWeekOrder) continue;							
				
				final int currentMatchWeekOrder=mw.getMatchWeekOrder()-1;			
				
				final LeagueSeasonRanking lsr=leagueSeasonRankingService.getLeagueSeasonRanking(league,currentYear,currentMatchWeekOrder);				
				for (Match m:mw.getMatchs())
				{	
					final MatchEvaluation matchEval=matchEvaluatorService.getMatchEvaluation(m);
					if (!matchEval.equals(MatchEvaluation.NOTPLAYED))
					{	
						try
						{
							final NeuralNetworkData nndInput=matchCodecService.getEncodedMatch(m,lsr);
							if (nndInput!=null)
							{
								final NeuralNetworkData nndExpectedOutput=matchEvaluationCodecService.getEncodedMatchEvaluation(matchEval);						
								inputs.add(nndInput);						
								expectedOutputs.add(nndExpectedOutput);
							}
						}
						catch(Exception e)
						{
							LOG.warn("Impossible to encode [l='"+league.getLeagueName()+"',ls='"+currentYear+"',mwo='"+mw.getMatchWeekOrder()+"',match='"+m+"']",e);						
						}
					}
				}
			}
			LOG.info("... training data set for season '"+currentYear+"' built, total size="+inputs.size()+" ...");									
		}
		LOG.info("... training data set built, total size="+inputs.size());		
		
		/* build prediction context */
		LOG.info("build prediction context ...");		
		final MatchPredictionContext mpc=new MatchPredictionContext(startSeasonYear,startMatchWeekOrder,endSeasonYear,endMatchWeekOrder,league);			
		mpc.setNeuralNetworksAggregate(getNeuralNetworksAggregateBuilderService().getNeuralNetworksAggregate(inputs,expectedOutputs));
		LOG.info("... prediction context built");	
		
		return mpc;
    }
}
