/*
 * Copyright (C) 2010 Olivier PARISOT <parisot_olivier@yahoo.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.doopyon.ravanelab.util;

import java.io.*;
import java.util.*;


/**
 * Repartition.
 *  
 * @author Olivier PARISOT
 */
public final class Repartition<E> implements Serializable 
{
	//
	// Static fields
	//
	
	/** Serial version UID. */
	private static final long serialVersionUID=1425047203742803521L;
	

	//
	// Instance fields
	//
	
	/** Internal map used to store the objects occurrence. */
	private final Map<E,Integer> internalMap;
	/** Internal counter used to store the object count. */
	private int internalSum;		
	
	
	//
	// Constructor
	//
	
	/**
	 * Constructor.
	 */
	public Repartition()
	{		
		this.internalMap=new HashMap<E,Integer>();		
		this.internalSum=0;			
	}
	
	
	//
	// Instance methods
	//
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() 
	{
		final StringBuilder sb=new StringBuilder();
		sb.append("\nRepartition:\n\n")
		  .append("\t").append("Classic:\n\n")
		  			   .append(/*toString(*/internalMap/*)*/)
		  			   .append("\n\n")
		  .append("\t").append("Sorted by value and normalized:\n\n")
		  			   .append(/*toString(*/getSortedByValueAndNormalizedMap()/*)*/)
		  			   .append("\n\n")
		  .append("\t").append("Sorted by key and normalized:\n\n")
		  			   .append(/*toString(*/getSortedByKeyAndNormalizedMap()/*)*/)
		  			   .append("\n\n");	  			   
		return sb.toString();
	}
	
	/**
	 * Build a String representation of a Map.
	 */
	/*private String toString(final Map<E,?> m)
	{
		final StringBuilder sb=new StringBuilder();
		sb.append("{");
		for (final E key:m.keySet())
		{
			sb.append("[").append(m.get(key)).append(",").append(key).append("] ");			
		}
		sb.append("}");
		return sb.toString();
	}*/
	
	/**
	 * Add an object into the repartition.
	 */
	public void addOccurenceFor(final E object)
	{
		if (!internalMap.containsKey(object)) internalMap.put(object,1);
		else internalMap.put(object,internalMap.get(object)+1);	    								
		internalSum++; 					
	}
	
	/**
	 * Add an object into the repartition.
	 */
	public int getOccurenceFor(final E object)
	{
		if (!internalMap.containsKey(object)) return 0;
		return internalMap.get(object);	    										 					
	}	
	
	/**
	 * 
	 * @return
	 */
	public Collection<E> getKeys()
	{
		return internalMap.keySet();
	}
	
	/**
	 * 
	 */
	public Map<E,Integer> getMap()
	{
		return Collections.unmodifiableMap(internalMap);
	}
	
	/**
	 * 
	 * @return
	 */
	public Map<E,Double> getSortedByValueAndNormalizedMap()
	{
		return buildSortedByValueAndNormalizedMap(internalMap,internalSum);
	} 	
	
	/**
	 * Commodity method used to sort and normalize a map.
	 */
	private Map<E,Double> buildSortedByValueAndNormalizedMap(final Map<E,Integer> map,final int sum)
	{
		final List<Map.Entry<E,Integer>> entriesList=new ArrayList<Map.Entry<E,Integer>>(map.entrySet());	 		  
		Collections.sort(entriesList,new MapEntryComparatorByValueAndOrderDesc<E>());		 				
		return normalizeAndRound(entriesList,sum);
	}	

	/**
	 * 
	 * @return
	 */
	public Map<E,Double> getSortedByKeyAndNormalizedMap()
	{
		return buildSortedByKeyAndNormalizedMap(internalMap,internalSum);
	} 	
	
	/**
	 * Commodity method used to sort and normalize a map.
	 */
	private Map<E,Double> buildSortedByKeyAndNormalizedMap(final Map<E,Integer> map,final int sum)
	{
		final List<Map.Entry<E,Integer>> entriesList=new ArrayList<Map.Entry<E,Integer>>(map.entrySet());	 		  
		Collections.sort(entriesList,new MapEntryComparatorByKeyAndOrderDesc<E>());		 				
		return normalizeAndRound(entriesList,sum);
	}	
	
	/**
	 * 
	 * @param entriesList
	 * @return
	 */
	private Map<E,Double> normalizeAndRound(final List<Map.Entry<E,Integer>> entriesList,final int sum)
	{
		final Map<E,Double> result=new LinkedHashMap<E,Double>();
		for (final Map.Entry<E,Integer> entry:entriesList) 
		{			
			result.put(entry.getKey(),MathsUtil.normalizeAndRound(entry.getValue().doubleValue(),(double)sum));
		}  		
		return result;		
	}
	
	/**
	 * Return the size of the repartition.
	 */
	public int getSize()
	{
		return internalMap.size();
	}
	
	/**
	 * 
	 * @return
	 */
	public String toStringShort()
	{
		return /*toString(*/getSortedByValueAndNormalizedMap().toString()/*)*/;
	}

	/**
	 * 
	 * @return
	 */
	public int getMaxOccurrence()
	{
		return MathsUtil.getMaxValue(internalMap.values());
	}
	
	/**
	 * 
	 * @return
	 */
	public E getTheMostOccurred() 
	{	
		final int maxOccur=getMaxOccurrence();
		for (Map.Entry<E,Integer> entry:internalMap.entrySet())
		{
			if (entry.getValue().intValue()==maxOccur) return entry.getKey();
		}
		return null;
	}		
	
	
	//
	// Internal classes
	//
	
	/**
	 * Map entry comparator, by value.
	 */	
	//@SuppressWarnings("hiding")
	private static class MapEntryComparatorByValueAndOrderDesc<E> implements Comparator<Map.Entry<E,Integer>>,Serializable 
	{
		/** Serial version UID. */
		private static final long serialVersionUID=1425047203742803515L;
		
		/**
		 * {@inheritDoc}
		 */
		@Override
		public int compare(final Map.Entry<E,Integer> e1,final Map.Entry<E,Integer> e2) 
	    {
	      return -e1.getValue().compareTo(e2.getValue());
	    }
	}

	/**
	 * Map entry comparator, by key.
	 */	
	//@SuppressWarnings("hiding")
	private static class MapEntryComparatorByKeyAndOrderDesc<E> implements Comparator<Map.Entry<E,Integer>>,Serializable 
	{
		/** Serial version UID. */
		private static final long serialVersionUID=1425047203742803512L;

		/**
		 * {@inheritDoc}
		 */
		@Override
		public int compare(final Map.Entry<E,Integer> e1,final Map.Entry<E,Integer> e2) 
	    {
	      return -e1.getKey().toString().compareTo(e2.getKey().toString());
	    }
	}


	
}
