/*
 * Copyright (C) 2010 Olivier PARISOT <parisot_olivier@yahoo.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.doopyon.ravanelab.util;

import java.util.*;


/**
 * Repartition delta.
 * 
 * @author Olivier PARISOT
 */
public final class RepartitionDelta<E> 
{
	//
	// Instance fields
	//

	/** */
	private final Repartition<E> before;
	/** */
	private final Repartition<E> after;
	/** */
	private final Map<E,Double> deltaMap;
	
	
	//
	// Constructor
	//
	
	/**
	 * Constructor
	 */
	public RepartitionDelta(final Repartition<E> before,final Repartition<E> after)
	{
		checkArguments(before,after);
		this.before=before;
		this.after=after;
		this.deltaMap=new HashMap<E,Double>(before.getSize());
		computeDelta0();
	}
	
	
	//
	// Instance methods
	//
	
	/**
	 * Check if the arguments are correct.	 
	 */
	private void checkArguments(final Repartition<E> before,final Repartition<E> after)
	{
		String errorMsg=null;
		if (before==null) errorMsg="First argument should not be null";
		if (after==null) errorMsg="Second argument should not be null";
		//if (before.getSize()!=after.getSize()) errorMsg="Both repartitions should have the same size! -> size(before)="+before.getSize()+" size(after)="+after.getSize();
		if (errorMsg!=null) throw new IllegalArgumentException(errorMsg);
	}
	
	/**
	 * Internal method used to compute the delta.
	 */
	private void computeDelta0()
	{
		final List<Map.Entry<E,Double>> listBefore=new ArrayList<Map.Entry<E,Double>>(before.getSortedByKeyAndNormalizedMap().entrySet());
		final List<Map.Entry<E,Double>> listAfter=new ArrayList<Map.Entry<E,Double>>(after.getSortedByKeyAndNormalizedMap().entrySet());
		final int sizeBefore=listBefore.size();
		final int sizeAfter=listAfter.size();
		for (int i=0;i<sizeBefore;i++)
		{
			boolean in1AndNotIn2=true;
			final Map.Entry<E,Double> entry1=listBefore.get(i);
			
			for (int j=0;j<sizeAfter;j++)
			{
				final Map.Entry<E,Double> entry2=listAfter.get(j);
				if (entry1.getKey().equals(entry2.getKey()))
				{
					final double val1=entry1.getValue();
					final double val2=entry2.getValue();
					if (!MathsUtil.areTheSame(val1,val2))
					{
						double diff;
						if (val1==0d) diff=1d;				
						else if (val2==0d) diff=-1d;				
						else diff=MathsUtil.normalizeAndRound(val2-val1,val1);
						deltaMap.put(entry1.getKey(),diff);
						in1AndNotIn2=false;
					}					
				}
			}
			if (in1AndNotIn2) deltaMap.put(entry1.getKey(),-1d);
		}
		
		for (int j=0;j<sizeAfter;j++)
		{
			final Map.Entry<E,Double> entry2=listAfter.get(j);
			boolean notIn1ButIn2=true;
			
			for (int i=0;i<sizeBefore;i++)
			{
				final Map.Entry<E,Double> entry1=listBefore.get(i);
				if (entry1.getKey().equals(entry2.getKey()))
				{
					notIn1ButIn2=false;
					break;
				}
			}
			
			if (notIn1ButIn2) deltaMap.put(entry2.getKey(),1d);
		}
	}
	
	/**
	 * 
	 * @param object
	 * @return
	 */
	public double getDelta(final E object)
	{
		if (!deltaMap.containsKey(object)) return 0d;
		else return deltaMap.get(object).doubleValue();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString()
	{
		final StringBuilder sb=new StringBuilder("** Delta between the two repartitions **\n\n");
		for (E o:deltaMap.keySet())
		{
			sb.append('\t').append(o).append(" -> ");
			final double val=deltaMap.get(o);
			if (val>=0) sb.append('+');
			sb.append(val).append("%\n");
		}
		return sb.toString();
	}
}
